package gov.va.med.mhv.sm.admin.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import gov.va.med.mhv.sm.admin.converter.TriageRelationConverter;
import gov.va.med.mhv.sm.admin.data.enums.SMErrorEnum;
import gov.va.med.mhv.sm.admin.dto.ClinicDTO;
import gov.va.med.mhv.sm.admin.dto.SMClinicDTO;
import gov.va.med.mhv.sm.admin.exception.SMApiException;
import gov.va.med.mhv.sm.admin.jpa.model.SMClinicsTriageMap;
import gov.va.med.mhv.sm.admin.jpa.model.TriageRelation;
import gov.va.med.mhv.sm.admin.jpa.repository.SMClinicsTriageMapRepository;
import gov.va.med.mhv.sm.admin.jpa.repository.TriageRelationRepository;
import gov.va.med.mhv.sm.admin.service.util.SMHealthshareDelegate;
import gov.va.med.mhv.sm.healthshare.wsclient.adminqueries.Clinic;
import gov.va.med.mhv.sm.healthshare.wsclient.adminqueries.SMClinic;
import gov.va.med.mhv.sm.admin.service.util.ResponseCodeUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.responses.ApiResponse;

@Path("/clinics")
@Service("clinicService")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
@Component
public class ClinicService{ 


    @Autowired
    private TriageRelationRepository triageRelationRepository;
	
    @Autowired
    private SMClinicsTriageMapRepository smClinicsTriageMapRepository;
    
    @Autowired
    private SMHealthshareDelegate smHealthShareDelegate;
    
    
	private static final Log log = LogFactory.getLog(ClinicService.class);

	@GET
	@Operation(summary = "Get Clinics based on Query Parameters provided.")
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200", 
	        			content = @Content(mediaType = "application/json", 
	        			array = @ArraySchema(schema = @Schema(implementation = ClinicDTO.class)))),
			@ApiResponse(responseCode = "404", description = "No clinics found matching query string",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Unexpected Error Occurred on getClinicsByTriageGroup",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})		
	public  List<ClinicDTO> getClinics(
									  @QueryParam("triageGroupId") Long triageGroupId,
									  @QueryParam("name") String clinicName,
		  							  @QueryParam("station") String station) throws SMApiException {
		try{
			ClinicEnum.SearchEnum clinicEnum = validateClinicsQueryParams(triageGroupId, clinicName, station);
			switch (clinicEnum) {
					case GET_CLINICS_BY_TRIAGE_GROUP_ID:
						return getClinicsByTriageGroup(triageGroupId);
					case GET_CLINICS_BY_NAME_AND_STATION:
						return getClinicsFromHSByNameAndStation(clinicName, station);
					default:
						throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
			}
		}catch(Exception clinicExp){
			log.error(clinicExp);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,clinicExp);
		}
	}
	
	private ClinicEnum.SearchEnum validateClinicsQueryParams(Long triageGroupId, String clinicName, String station) throws SMApiException{
		if(triageGroupId !=null){
			if(clinicName!=null || station!=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else{
				return ClinicEnum.SearchEnum.GET_CLINICS_BY_TRIAGE_GROUP_ID;
			}
		}else if(station!=null){
			if(triageGroupId !=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else if(clinicName !=null){
						return ClinicEnum.SearchEnum.GET_CLINICS_BY_NAME_AND_STATION;
				  }else{
						throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
				  }
		}
		else{
			throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
		}
	}
	
	@Path("/smclinics")
	@GET
	@Operation(summary = "Get SMClinics based on Query Parameters provided.")
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200", 
	        			content = @Content(mediaType = "application/json", 
	        			array = @ArraySchema(schema = @Schema(implementation = SMClinicDTO.class)))),
			@ApiResponse(responseCode = "404", description = "No SM Clinics found matching query string",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Unexpected Error Occurred on getClinicsByTriageGroup",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})		
	public  List<SMClinicDTO> getSMClinics(
									  @QueryParam("triageGroupId") Long triageGroupId,
									  @QueryParam("name") String smClinicName,
		  							  @QueryParam("station") String station) throws SMApiException {
		try{
			ClinicEnum.SearchEnum clinicEnum = validateSMClinicsQueryParams(triageGroupId, smClinicName, station);
			switch (clinicEnum) {
					case GET_SMCLINICS_BY_TRIAGE_GROUP_ID:
						return getSMClinicsByTriageGroup(triageGroupId);
					case GET_SMCLINICS_BY_NAME_AND_STATION:
						return getSMClinicsFromHSByNameAndStation(smClinicName, station);
					default:
						throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
			}
		}catch(Exception smClinicExp){
			log.error(smClinicExp);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,smClinicExp);
		}
	}
	

	private List<ClinicDTO> getClinicsByTriageGroup(Long triageGroupId)  throws SMApiException {
		List<ClinicDTO> clinicList = null;
		List<TriageRelation> triageRelationList = null;
		try 
		{	if(triageGroupId != null) {
				triageRelationList = triageRelationRepository.getClinicsByTriageGroup(triageGroupId);
				if(triageRelationList != null){
					clinicList = TriageRelationConverter.convertClinicList(triageRelationList);
				}else {
					throw new SMApiException(SMErrorEnum.CLINICS_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
				}
				
			}

		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e);
		}

		return clinicList;
	}
	
	private List<ClinicDTO> getClinicsFromHSByNameAndStation(String name, String station) throws SMApiException {
		List<ClinicDTO> clinicDtoList = null;	
		List<Clinic> hsClinicList = null;
		try{
			if(name != null || station !=null) {
				hsClinicList = smHealthShareDelegate.getClinicsByName(name,station);
				if(hsClinicList !=null){
					clinicDtoList = TriageRelationConverter.convertHSClinicList(hsClinicList);
				}else{
					throw new SMApiException("No clinics found matching query string", ResponseCodeUtil.STATUS_NOT_FOUND);
				}
			}
		}catch(Exception e4){
			if(log.isErrorEnabled()){
				log.error("ClinicServiceImpl=>Error Occured on getClinicsFromHSByNameAndStation");
				e4.printStackTrace();
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e4);
			}
		}
		return clinicDtoList;
	}
	
	private ClinicEnum.SearchEnum validateSMClinicsQueryParams(Long triageGroupId, String smClinicName, String station) throws SMApiException{
		if(triageGroupId !=null){
			if(smClinicName!=null || station!=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else{
				return ClinicEnum.SearchEnum.GET_SMCLINICS_BY_TRIAGE_GROUP_ID;
			}
		}else if(station!=null){
			if(triageGroupId !=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else if(smClinicName !=null){
						return ClinicEnum.SearchEnum.GET_SMCLINICS_BY_NAME_AND_STATION;
				  }else{
						throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
				  }
		}
		else{
			throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
		}
	}
				
	private List<SMClinicDTO> getSMClinicsFromHSByNameAndStation(String smClinicName, String station) throws SMApiException  {
		List<SMClinicDTO> smClinicDtoList = null;
		List<SMClinic> hsSMClinicList = null;
		try{
			if(smClinicName != null && station !=null) {
				hsSMClinicList = smHealthShareDelegate.getSMClinicsByName(smClinicName, station);
				if(hsSMClinicList !=null){
					smClinicDtoList = TriageRelationConverter.convertHSSmClinicList(hsSMClinicList);
				}else{
					throw new SMApiException(SMErrorEnum.CLINICS_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
				}
			}
		}catch(Exception e4){
			if(log.isErrorEnabled()){
				log.error("ClinicServiceImpl=>Error Occured on getClinicsFromHSByNameAndStation");
				e4.printStackTrace();
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e4);
			}
		}
		return smClinicDtoList;
	}
	
	private List<SMClinicDTO> getSMClinicsByTriageGroup(Long triageGroupId) throws SMApiException  {
		List<SMClinicDTO> smClinicDtoList = null; 
		SMClinicsTriageMap smClinicsTriageMap = null;
		try{
			if(triageGroupId !=null){
				smClinicsTriageMap = smClinicsTriageMapRepository.getSMClinicsByTriageGroup(triageGroupId);
				if(smClinicsTriageMap !=null){
					smClinicDtoList = TriageRelationConverter.convertSMClinicList(smClinicsTriageMap);
				}
			}
		}catch(Exception e4){
			if(log.isErrorEnabled()){
				log.error("ClinicServiceImpl=>Error Occured on getSMClinicsByTriageGroup");
				e4.printStackTrace();
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e4);
			}
		}
		return smClinicDtoList;
	}
	
	@GET
	@Path("/smclinics/tiutitles")
	@Operation(summary = "Get TIU Titles from Health Share")
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200", 
	        			content = @Content(mediaType = "application/json", 
	        			array = @ArraySchema(schema = @Schema(implementation = String.class)))),
			@ApiResponse(responseCode = "404", description = "CPRSTiuTitles are not found",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Unexpected Error Occurred",content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})				
	public List<String> getCPRSTiuTitlesFromHSByStation(@NotNull @QueryParam("station") String station) throws SMApiException {
		List<String> titTitleList = new ArrayList<String>();		
		try{
				smHealthShareDelegate.getTiuNoteTitlesByFacility(station).stream().forEach((TiuTitles)->{
				titTitleList.add(TiuTitles.getTIUTitle());
				});
				
				if(titTitleList == null || titTitleList.size()==0) {
					throw new SMApiException(SMErrorEnum.CPRS_TIT_TILES_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
				}
		}catch(Exception e4){
			if(log.isErrorEnabled()){
				log.error("ClinicServiceImpl=>Error Occured on getCPRSTiuTitlesFromHSByStation");
				e4.printStackTrace();
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e4);
			}
		}
		return titTitleList;
	}
	
	private static class ClinicEnum{
		public enum SearchEnum{
			GET_CLINICS_BY_TRIAGE_GROUP_ID,
			GET_CLINICS_BY_NAME_AND_STATION,
			GET_SMCLINICS_BY_TRIAGE_GROUP_ID,
			GET_SMCLINICS_BY_NAME_AND_STATION
		}
	}
	
	
}
